unit Dsmatch;
{
  String Comparison Class
  Copyright (c) 1996, by Philip Stevenson
}

interface

uses
  SysUtils, WinTypes, WinProcs;

type

  { TCompareTextLines class}

  TCompareTextLines = class
  private
    { Private declarations }
    FS1: string;
    FS2: string;
    FExcludeLeadingWhitespace: boolean;
    FCaseSensitive: boolean;
    FMatch: boolean;
  public
    { Public declarations }
    constructor Create(AExcludeLeadingWhitespace,
      ACaseSensitive: boolean);
    procedure SetStrings(const AS1, AS2: string);
    function StrMatch(const AS1, AS2: string): boolean;
    function MatchAtPos: boolean;
    function NearlySame: boolean;

    property ExactMatch: boolean read FMatch;
    property IgnoreLeadingWhitespace: boolean read FExcludeLeadingWhitespace;
    property IgnoreCase: boolean read FCaseSensitive;
  end;

  function TrimR(S: string): string;
  function TrimL(S: string): string;

implementation

  { Ubiquitous Trim functions}

  function TrimR(S: string): string;
  begin
    while (Length(S) > 0) and (S[Length(S)] <= ' ') do
      Delete(S, Length(S), 1);
    Result := S;
  end;

  function TrimL(S: string): string;
  begin
    while (Length(S) > 0) and (S[1] <= ' ') do
      Delete(S, 1, 1);
    Result := S;
  end;

  { TCompareTextLines methods}

constructor TCompareTextLines.Create(AExcludeLeadingWhitespace,
  ACaseSensitive: boolean);
begin
  inherited Create;
  FExcludeLeadingWhitespace := AExcludeLeadingWhitespace;
  FCaseSensitive := ACaseSensitive;
end;

procedure TCompareTextLines.SetStrings(const AS1, AS2: string);
{-Two strings to mess with }
begin
  FS1 := AS1;
  FS2 := AS2;
  FMatch := FS1 = FS2;
end;

function TCompareTextLines.StrMatch(const AS1, AS2: string): boolean;
{-Compare two strings for equal}
var
  S1, S2: string;
begin
  {Delete trailing spaces}
  S1 := TrimR(AS1);
  S2 := TrimR(AS2);
  if FExcludeLeadingWhitespace then {Delete leading spaces}
  begin
    S1 := TrimL(S1);
    S2 := TrimL(S2);
  end;
  if FCaseSensitive then
    Result := CompareStr(S1, S2) = 0
  else
    Result := CompareText(S1, S2) = 0;
end;

function TCompareTextLines.MatchAtPos: boolean;
begin
  Result := StrMatch(FS1, FS2);
end;

function TCompareTextLines.NearlySame:boolean;
{-Determine if two strings are a lot alike}
const
  MIN_MUST_MATCH = 60; {%}
var
  hits, width, samples: Smallint;
  frag: string[50];
  Sa, Sb, Sw: string;
  len: Integer;
begin
  Result := False;
  Sa := TrimL(FS1);
  Sb := TrimL(FS2);
  { make set from longer line}
  if Length(Sb) >= Length(Sa) then {swap}
  begin
    Sw := Sb;
    Sb := Sa;
    Sa := Sw;
  end;
  width := Length(Sa) div 5;
  if width < 1 then
    width := 1;
  if width > 5 then
    width := 5;
  samples := 0;
  hits := 0;

  { Take samples, count matches}
  repeat
    frag := System.Copy(Sa, (samples*width)+1, width);
    inc(samples);
    len := pos(frag, Sb);
    if len <> 0 then
    begin
      inc(hits);
      System.Delete(Sb, len, width); {so it won't match again}
    end
  until frag = '';
  dec(samples);
  if samples <= 0 then {null strings are no match}
    exit;
  FMatch := hits = samples;
  if FMatch then
    Result := FMatch
  else
    Result := ((hits*100) div samples) >= MIN_MUST_MATCH;
end;

end.
